﻿// 2011 IDesign Inc.
// Pytania? Komentarze? Odwiedź
// http://www.idesign.net

using System;
using System.Collections;
using System.Collections.Generic;

namespace ServiceModelEx
{
   public static class Collection
   {
      /// <summary>
      /// Pomija elementy kolekcji do momentu spełnienia określonego warunku,
      /// po czym zwraca pozostałe elementy.
      /// </summary>
      public static IEnumerable<T> SkipWhile<T>(IEnumerable<T> collection,Predicate<T> match)
      {
         if(collection == null)
         {
            throw new ArgumentNullException("collection");
         }
         if(match == null)
         {
            throw new ArgumentNullException("match");
         }
         foreach(T item in collection)
         {
            if(match(item) == false)
            {
               yield return item;
            }
         }
      }

      /// <summary>
      /// Zwraca określoną liczbę kolejnych elementów z początku kolekcji.
      /// </summary>
      public static IEnumerable<T> Take<T>(IEnumerable<T> collection,int count)
      {
         if(collection == null)
         {
            throw new ArgumentNullException("collection");
         }
         if(count < 0)
         {
            throw new IndexOutOfRangeException("count");
         }
         int i = 0;
         foreach(T item in collection)
         {
            i++;
            if(i > count)
            {
               yield return item;
            }
         }
      }

      /// <summary>
      /// Zwraca kolejne elementy kolekcji dopóki określony warunek jest spełniony.
      /// </summary>
      /// <typeparam name="T"></typeparam>
      /// <param name="collection"></param>
      /// <param name="match"></param>
      /// <returns>Obiekt typu System.Collections.Generic.IEnumerable<T> zawierający elementy kolekcji
      ///     wejściowej występujące przed pierwszym elementem, dla którego wskazany warunek nie jest
      ///     spełniony (ma wartość false).</returns>
      public static IEnumerable<T> TakeWhile<T>(IEnumerable<T> collection,Predicate<T> match)
      {
         if(collection == null)
         {
            throw new ArgumentNullException("collection");
         }
         if(match == null)
         {
            throw new ArgumentNullException("match");
         }
         foreach(T item in collection)
         {
            if(match(item))
            {
               yield break;
            }
            yield return item;
         }
      }
      /// <summary>
      /// Określa, czy dwie kolekcje są sobie równe (porównuje kolejne elementy tych kolekcji).
      /// </summary>
      public static bool SequenceEqual<T>(IEnumerable<T> first,IEnumerable<T> second) where T : IEquatable<T>
      {
         if(first == null && second == null)
         {
            return true;
         }
         if(first ==  null)
         {
            throw new ArgumentNullException("first");
         }
         if(second ==  null)
         {
            throw new ArgumentNullException("second");
         }
         if(Count(first) != Count(second))
         {
            return false;
         }
         using(IEnumerator<T> enumerator1 = first.GetEnumerator())
         using(IEnumerator<T> enumerator2 = second.GetEnumerator())
         {

            int count = Count(first);

            int index = 1;
            while(enumerator1.Current.Equals(enumerator2.Current))
            {
               index++;
               continue;
            }
            return index == count;
         }
      }
      /// <summary>
      /// Pomija określoną liczbę elementów kolekcji i zwraca pozostałe elementy.
      /// </summary>
      public static IEnumerable<T> Skip<T>(IEnumerable<T> collection,int count)
      {
         if(collection == null)
         {
            throw new ArgumentNullException("collection");
         }
         if(count < 0)
         {
            throw new IndexOutOfRangeException("count");
         }
         int i = 0;
         foreach(T item in collection)
         {
            i++;
            if(i > count)
            {
               yield return item;
            }
         }
      }


      /// <summary>
      /// Generuje kolekcję złożoną z jednej, wielokrotnie powtórzonej wartości.
      /// </summary>
      public static IEnumerable<T> Repeat<T>(T element,int count)
      {
         if(count <= 0)
         {
            throw new ArgumentOutOfRangeException("count");
         }
         for(int i = 0;i< count;i++)
         {
            yield return element;
         }
      }
      /// <summary>
      /// Zwraca element znajdujący się na określonej pozycji w kolekcji.
      /// </summary>
      /// <typeparam name="T"></typeparam>
      /// <param name="collection">Obiekt typu System.Collections.Generic.IEnumerable<T>, z którego zostanie odczytany element.</param>
      /// <param name="index">Indeks żądanego elementu (indeks pierwszego elementu jest równy 0).</param>
      /// <returns></returns>
      public static T ElementAt<T>(IEnumerable<T> collection,int index)
      {
         if(collection == null)
         {
            throw new ArgumentNullException("collection");
         }
         if(index < 0 || index > Count(collection) -1)
         {
            throw new ArgumentOutOfRangeException("index");
         }
         return ToArray(collection)[index];
      }

      /// <summary>
      /// Zwraca element znajdujący się na określonej pozycji w kolekcji.
      /// </summary>
      /// <typeparam name="T"></typeparam>
      /// <param name="collection">Obiekt typu System.Collections.Generic.IEnumerable<T>, z którego zostanie odczytany element.</param>
      /// <param name="index">Indeks żądanego elementu (indeks pierwszego elementu jest równy 0).</param>
      /// <returns></returns>
      public static T ElementAtOrDefault<T>(IEnumerable<T> collection,int index)
      {
         if(index < 0 || index > Count(collection) -1 || collection == null)
         {
            return default(T);
         }
         return ElementAt(collection,index);
      }
      /// <summary>
      /// Konkatenuje dwie kolekcje.
      /// </summary>
      public static IEnumerable<T> Concat<T>(IEnumerable<T> first,IEnumerable<T> second)
      {
         if(first == null)
         {
            throw new ArgumentNullException("first");
         }
         if(second == null)
         {
            throw new ArgumentNullException("second");
         }
         foreach(T item in first)
         {
            yield return item;
         }
         foreach(T item in second)
         {
            yield return item;
         }
      }
      /// <summary>
      /// Zwraca wartość true, jeśli kolekcja zawiera dany element.
      /// </summary>
      /// <typeparam name="T"></typeparam>
      /// <param name="collection"></param>
      /// <param name="item"></param>
      /// <returns></returns>
      public static bool Contains<T>(IEnumerable<T> collection,T item) where T : IEquatable<T>
      {
         if(collection == null)
         {
            throw new ArgumentNullException("collection");
         }
         foreach(T t in collection)
         {
            if(t.Equals(item))
            {
               return true;
            }
         }
         return false;
      }
      /// <summary>
      /// Konwertuje wszystkie elementy typu T w kolekcji na nową kolekcję typu U (zgodnie z użytym konwerterem).
      /// </summary>
      /// <typeparam name="T"></typeparam>
      /// <typeparam name="U"></typeparam>
      /// <param name="collection"></param>
      /// <param name="converter"></param>
      /// <returns></returns>
      public static IEnumerable<U> ConvertAll<T,U>(IEnumerable<T> collection,Converter<T,U> converter)
      {
         if(collection == null)
         {
            throw new ArgumentNullException("collection");
         }
         if(converter == null)
         {
            throw new ArgumentNullException("converter");
         }
         foreach(T item in collection)
         {
            yield return converter(item);
         }
      }

      /// <summary>
      /// Zwraca wartość true, jeśli dana kolekcja zawiera element zgodny z przekazanym predykatem.
      /// </summary>
      /// <typeparam name="T"></typeparam>
      /// <param name="collection"></param>
      /// <param name="match"></param>
      /// <returns></returns>
      public static bool Any<T>(IEnumerable<T> collection,Predicate<T> match)
      {
         if(collection == null)
         {
            throw new ArgumentNullException("collection");
         }
         if(match == null)
         {
            throw new ArgumentNullException("match");
         }
         foreach(T item in collection)
         {
            if(match(item))
            {
               return true;
            }
         }
         return false;
      }
      /// <summary>
      /// Określa, czy kolekcja zawiera przynajmniej jeden element.
      /// </summary>
      /// <typeparam name="T"></typeparam>
      /// <param name="collection"></param>
      /// <param name="match"></param>
      /// <returns></returns>
      public static bool Any<T>(IEnumerable<T> collection)
      {
         if(collection == null)
         {
            throw new ArgumentNullException("collection");
         }
         using(IEnumerator<T> iterator = collection.GetEnumerator())
         {
            return iterator.MoveNext();
         }
      }
      /// <summary>
      /// Zwraca jedyny element danej kolekcji jednoelementowej.
      /// </summary>
      /// <typeparam name="T"></typeparam>
      /// <param name="collection"></param>
      /// <param name="match"></param>
      /// <returns></returns>
      public static T Single<T>(IEnumerable<T> collection)
      {
         if(collection == null)
         {
            throw new ArgumentNullException("collection");
         }
         if(Count(collection) != 1)
         {
            throw new InvalidOperationException("collection");
         }
         return ToList(collection)[0];
      }


      /// <summary>
      /// Zwraca jedyny element kolekcji zgodny z przekazanym predykatem.
      /// </summary>
      /// <typeparam name="T"></typeparam>
      /// <param name="collection"></param>
      /// <param name="match"></param>
      /// <returns></returns>
      public static T Single<T>(IEnumerable<T> collection,Predicate<T> match)
      {
         if(collection == null)
         {
            throw new ArgumentNullException("collection");
         }
         if(match == null)
         {
            throw new ArgumentNullException("match");
         }
         if(Count(collection) == 0)
         {
            throw new InvalidOperationException("collection");
         }
         IList<T> list = ToList(FindAll(collection,match));
         if(list.Count != 1)
         {
            throw new InvalidOperationException("Multiple matches found");
         }
         return list[0];
      }



      /// <summary>
      /// Zwraca jedyny element danej kolekcji lub element domyślny.
      /// </summary>
      /// <typeparam name="T"></typeparam>
      /// <param name="collection"></param>
      /// <param name="match"></param>
      /// <returns></returns>
      public static T SingleOrDefault<T>(IEnumerable<T> collection)
      {
         if(collection == null || Count(collection) != 1)
         {
            return default(T);
         }
         return Single(collection);
      }


      /// <summary>
      /// Zwraca jedyny element danej kolekcji zgodny z przekazanym predykatem.
      /// </summary>
      /// <typeparam name="T"></typeparam>
      /// <param name="collection"></param>
      /// <param name="match"></param>
      /// <returns></returns>
      public static T SingleOrDefault<T>(IEnumerable<T> collection,Predicate<T> match)
      {
         if(collection == null || match == null)
         {
            return default(T);
         }
         if(Count(collection) == 0)
         {
            return default(T);
         }
         IList<T> list = ToList(FindAll(collection,match));
         if(list.Count != 1)
         {
            return default(T);
         }
         return list[0];
      }
      /// <summary>
      /// Zwraca pierwszy element kolekcji.
      /// </summary>
      /// <typeparam name="T"></typeparam>
      /// <param name="collection"></param>
      /// <param name="match"></param>
      /// <returns></returns>
      public static T First<T>(IEnumerable<T> collection)
      {
         if(collection == null)
         {
            throw new ArgumentNullException("collection");
         }
         if(Count(collection) == 0)
         {
            throw new InvalidOperationException("collection");
         }
         return ToList(collection)[0];
      }

      /// <summary>
      /// Zwraca pierwszy element kolekcji.
      /// </summary>
      /// <typeparam name="T"></typeparam>
      /// <param name="collection"></param>
      /// <param name="match"></param>
      /// <returns></returns>
      public static T FirstOrDefault<T>(IEnumerable<T> collection)
      {
         if(Count(collection) == 0 || collection == null)
         {
            return default(T);
         }
         return ToList(collection)[0];
      }

      /// <summary>
      /// Zwraca pierwsze wystąpienie elementu kolekcji zgodnego z przekazanym predykatem.
      /// </summary>
      /// <typeparam name="T"></typeparam>
      /// <param name="collection"></param>
      /// <param name="match"></param>
      /// <returns></returns>
      public static T First<T>(IEnumerable<T> collection,Predicate<T> match)
      {
         if(collection == null)
         {
            throw new ArgumentNullException("collection");
         }
         if(match == null)
         {
            throw new ArgumentNullException("match");
         }
         foreach(T item in collection)
         {
            if(match(item))
            {
               return item;
            }
         }
         return default(T);
      }
      /// <summary>
      /// Znajduje wszystkie elementy kolekcji zgodne z przekazanym predykatem.
      /// </summary>
      /// <typeparam name="T"></typeparam>
      /// <param name="collection"></param>
      /// <param name="match"></param>
      /// <returns></returns>
      public static IEnumerable<T> FindAll<T>(IEnumerable<T> collection,Predicate<T> match)
      {
         if(collection == null)
         {
            throw new ArgumentNullException("collection");
         }
         if(match == null)
         {
            throw new ArgumentNullException("match");
         }
         foreach(T item in collection)
         {
            if(match(item))
            {
               yield return item;
            }
         }
      }
      /// <summary>
      /// Zwraca wszystkie elementy iteratora iterator1, które nie występują w kolekcji collection2
      /// </summary>
      /// <typeparam name="T"></typeparam>
      /// <param name="iterator1"></param>
      /// <param name="collection2"></param>
      /// <returns></returns>
      public static IEnumerable<T> Complement<T>(IEnumerable<T> collection1,IEnumerable<T> collection2) where T : IEquatable<T>
      {
         foreach(T item in collection1)
         {
            if(Contains(collection2,item) == false)
            {
               yield return item;
            }
         }
      }

      /// <summary>
      /// Zwraca wszystkie elementy spoza ilocznu (części wspólnej) kolekcji collection1 i collection2.
      /// </summary>
      /// <typeparam name="T"></typeparam>
      /// <param name="collection1"></param>
      /// <param name="collection2"></param>
      /// <returns></returns>
      public static IEnumerable<T> Except<T>(IEnumerable<T> collection1,IEnumerable<T> collection2) where T : IEquatable<T>
      {
         IEnumerable<T> complement1 = Complement(collection1,collection2);
         IEnumerable<T> complement2 = Complement(collection2,collection1);
         return Union(complement1,complement2);
      }
      /// <summary>
      /// Zwraca kolekcję obejmującą wszystkie unikatowe elementy przekazanej kolekcji (bez powtórzeń)
      /// </summary>
      /// <typeparam name="T"></typeparam>
      /// <param name="collection"></param>
      /// <returns></returns>
      public static IEnumerable<T> Distinct<T>(IEnumerable<T> collection)
      {
         if(collection == null)
         {
            throw new ArgumentNullException("collection");
         }
         IList<T> list = new List<T>();
         foreach(T item in collection)
         {
            if(list.Contains(item) == false)
            {
               list.Add(item);
            }
         }
         foreach(T item in list)
         {
            yield return item;
         }
      }
      /// <summary>
      /// Zwraca indeks pierwszego wystąpienia elementu w kolekcji.
      /// </summary>
      /// <typeparam name="T"></typeparam>
      /// <param name="collection"></param>
      /// <param name="value"></param>
      /// <returns></returns>
      public static int FindIndex<T>(IEnumerable<T> collection,T value) where T : IEquatable<T>
      {
         if(collection == null)
         {
            throw new ArgumentNullException("collection");
         }
         int index = 0;

         foreach(T item in collection)
         {
            if(item.Equals(value) == false)
            {
               index++;
            }
            else
            {
               return index;
            }
         }
         return -1;
      }
      /// <summary>
      /// Zwraca część wspólną kolekcji collection1 i collection2
      /// </summary>
      /// <typeparam name="T"></typeparam>
      /// <param name="collection1"></param>
      /// <param name="collection2"></param>
      /// <returns></returns>
      public static IEnumerable<T> Intersect<T>(IEnumerable<T> collection1,IEnumerable<T> collection2) where T : IEquatable<T>
      {
         Predicate<T> existsInCollection2 = delegate(T t)
                                            {
                                               Predicate<T> exist = delegate(T item)
                                                                    {
                                                                       return item.Equals(t);
                                                                    };
                                               return Any(collection2,exist);
                                            };
         return FindAll(collection1,existsInCollection2);
      }
      /// <summary>
      /// Zwraca indeks ostatniego wystąpienia w kolekcji elementu zgodnego z przekazanym predykatem
      /// </summary>
      /// <typeparam name="T"></typeparam>
      /// <param name="collection"></param>
      /// <param name="match"></param>
      /// <returns></returns>
      public static T LastOrDefault<T>(IEnumerable<T> collection,Predicate<T> match)
      {
         if(collection == null)
         {
            return default(T);
         }
         if(Count(collection) == 0)
         {
            return default(T);
         }
         if(match == null)
         {
            throw new ArgumentNullException("match");
         }
         return Last(collection,match);
      }

      /// <summary>
      /// Zwraca indeks ostatniego wystąpienia elementu w kolekcji
      /// </summary>
      /// <typeparam name="T"></typeparam>
      /// <param name="collection"></param>
      /// <param name="match"></param>
      /// <returns></returns>
      public static T LastOrDefault<T>(IEnumerable<T> collection)
      {
         if(collection == null)
         {
            return default(T);
         }
         if(Count(collection) == 0)
         {
            return default(T);
         }
         return Last(collection);
      }
      /// <summary>
      /// FiZwraca indeks ostatniego wystąpienia elementu w kolekcji
      /// </summary>
      /// <typeparam name="T"></typeparam>
      /// <param name="collection"></param>
      /// <param name="match"></param>
      /// <returns></returns>
      public static T Last<T>(IEnumerable<T> collection)
      {
         if(collection == null)
         {
            throw new ArgumentNullException("collection");
         }
         return ToArray(collection)[Count(collection)-1];
      }

      /// <summary>
      /// Zwraca indeks ostatniego wystąpienia w kolekcji elementu zgodnego z przekazanym predykatem
      /// </summary>
      /// <typeparam name="T"></typeparam>
      /// <param name="collection"></param>
      /// <param name="match"></param>
      /// <returns></returns>
      public static T Last<T>(IEnumerable<T> collection,Predicate<T> match)
      {
         if(collection == null)
         {
            throw new ArgumentNullException("collection");
         }
         if(match == null)
         {
            throw new ArgumentNullException("match");
         }
         T last = default(T);

         foreach(T item in collection)
         {
            if(match(item))
            {
               last = item;
            }
         }
         return last;
      }
      /// <summary>
      /// Zwraca indeks ostatniego wystąpienia elementu w kolekcji
      /// </summary>
      /// <typeparam name="T"></typeparam>
      /// <param name="collection"></param>
      /// <param name="value"></param>
      /// <returns></returns>
      public static int FindLastIndex<T>(IEnumerable<T> collection,T value) where T : IEquatable<T>
      {
         if(collection == null)
         {
            throw new ArgumentNullException("collection");
         }
         int last = -1;

         foreach(T item in collection)
         {
            if(item.Equals(value))
            {
               last++;
            }
         }
         if(last >= 0)
         {
            return last;
         }
         else
         {
            return -1;
         }
      }
      /// <summary>
      /// Zwraca sumę kolekcji collection1 i collection2
      /// </summary>
      /// <typeparam name="T"></typeparam>
      /// <param name="collection1"></param>
      /// <param name="collection2"></param>
      /// <returns></returns>
      public static IEnumerable<T> Union<T>(IEnumerable<T> collection1,IEnumerable<T> collection2) where T : IEquatable<T>
      {
         LinkedList<T> union = new LinkedList<T>(Distinct(collection1));

         Action<T> action =   delegate(T t)
                              {
                                 if(union.Find(t) == null)
                                 {
                                    union.AddLast(t);
                                 }
                              };
         ForEach(collection2,action);
         foreach(T item in union)
         {
            yield return item;
         }
      }
      /// <summary>
      /// Wykonuje przekazaną akcję dla wszystkich elementów w kolekcji
      /// </summary>
      /// <typeparam name="T"></typeparam>
      /// <param name="collection"></param>
      /// <param name="action"></param>
      public static void ForEach<T>(IEnumerable<T> collection,Action<T> action)
      {
         if(collection == null)
         {
            throw new ArgumentNullException("collection");
         }
         if(action == null)
         {
            throw new ArgumentNullException("action");
         }
         foreach(T item in collection)
         {
            action(item);
         }
      }
      /// <summary>
      /// Zwraca liczbę elementów kolekcji spełniających przekazany warunek
      /// </summary>
      /// <typeparam name="T"></typeparam>
      /// <param name="collection"></param>
      /// <returns></returns>
      public static int Count<T>(IEnumerable<T> collection,Predicate<T> match)
      {
         int length = 0;
         Action<T> add = delegate(T item)
                         {
                            if(match(item))
                            {
                               length++;
                            }
                         };
         ForEach(collection,add);
         return length;
      }
      /// <summary>
      /// Zwraca liczbę elementów kolekcji
      /// </summary>
      /// <typeparam name="T"></typeparam>
      /// <param name="collection"></param>
      /// <returns></returns>
      public static int Count<T>(IEnumerable<T> collection)
      {
         int length = 0;
         Action<T> add = delegate
                         {
                            length++;
                         };
         ForEach(collection,add);
         return length;
      }
      /// <summary>
      /// Zwraca kolekcję ze wszystkimi elementami w odwrotnej kolejności
      /// </summary>
      /// <typeparam name="T"></typeparam>
      /// <param name="collection"></param>
      /// <returns></returns>
      public static IEnumerable<T> Reverse<T>(IEnumerable<T> collection)
      {
         if(collection == null)
         {
            throw new ArgumentNullException("collection");
         }
         List<T> list = new List<T>(collection);
         list.Reverse();

         foreach(T item in list)
         {
            yield return item;
         }
      }
      /// <summary>
      /// Sortuje kolekcję
      /// </summary>
      /// <typeparam name="T"></typeparam>
      /// <param name="collection"></param>
      /// <returns></returns>
      public static IEnumerable<T> Sort<T>(IEnumerable<T> collection)
      {
         if(collection == null)
         {
            throw new ArgumentNullException("collection");
         }
         List<T> list = new List<T>(collection);
         list.Sort();

         foreach(T item in list)
         {
            yield return item;
         }
      }
      /// <summary>
      /// Konwertuje kolekcję na tablicę
      /// </summary>
      /// <typeparam name="T"></typeparam>
      /// <param name="collection"></param>
      /// <returns></returns>
      public static T[] ToArray<T>(IEnumerable<T> collection)
      {
         if(collection == null)
         {
            throw new ArgumentNullException("collection");
         }
         List<T> list = new List<T>();

         foreach(T item in collection)
         {
            list.Add(item);
         }
         return list.ToArray();
      }

      /// <summary>
      /// Konwertuje iterator na tablicę
      /// </summary>
      /// <typeparam name="T"></typeparam>
      /// <param name="iterator"></param>
      /// <returns></returns>
      public static T[] ToArray<T>(IEnumerator<T> iterator)
      {
         if(iterator == null)
         {
            throw new ArgumentNullException("iterator");
         }
         List<T> list = new List<T>();

         while(iterator.MoveNext())
         {
            list.Add(iterator.Current);
         }
         return list.ToArray();
      }
      /// <summary>
      /// Konwertuje elementy kolekcji na tablicę
      /// </summary>
      /// <typeparam name="T"></typeparam>
      /// <param name="iterator"></param>
      /// <param name="count">Początkowy rozmiar optymalizacji</param>
      /// <returns></returns>
      public static T[] ToArray<T>(IEnumerable<T> collection,int count)
      {
         if(collection == null)
         {
            throw new ArgumentNullException("iterator");
         }
         return ToArray(collection.GetEnumerator(),count);
      }
      /// <summary>
      /// Konwertuje elementy iteratora na tablicę
      /// </summary>
      /// <typeparam name="T"></typeparam>
      /// <param name="iterator"></param>
      /// <param name="count">Początkowy rozmiar optymalizacji</param>
      /// <returns></returns>
      public static T[] ToArray<T>(IEnumerator<T> iterator,int count)
      {
         if(iterator == null)
         {
            throw new ArgumentNullException("iterator");
         }
         List<T> list = new List<T>(count);

         while(iterator.MoveNext())
         {
            list.Add(iterator.Current);
         }

         return list.ToArray();
      }
      /// <summary>
      /// Konwertuje elementy kolekcji na tablicę, której typ zależy od użytego konwertera
      /// </summary>
      /// <typeparam name="T"></typeparam>
      /// <typeparam name="U"></typeparam>
      /// <param name="collection"></param>
      /// <param name="converter"></param>
      /// <returns></returns>
      static U[] ToArray<T,U>(IEnumerable<T> collection,Converter<T,U> converter)
      {
         int count = Count(collection);
         return ToArray(collection,converter,count);
      }
      /// <summary>
      /// Konwertuje elementy kolekcji na tablicę typu U (zależnie od użytego konwertera)
      /// </summary>
      /// <typeparam name="T"></typeparam>
      /// <param name="iterator"></param>
      /// <param name="count">Initial size for optimization</param>
      /// <returns></returns>
      static U[] ToArray<T,U>(IEnumerable<T> collection,Converter<T,U> converter,int count)
      {
         List<U> list = new List<U>(count);
         foreach(T t in collection)
         {
            list.Add(converter(t));
         }
         return list.ToArray();
      }
      /// <summary>
      /// Zwraca listę złożoną z elementów kolekcji
      /// </summary>
      /// <typeparam name="T"></typeparam>
      /// <param name="collection"></param>
      /// <returns></returns>
      public static IList<T> ToList<T>(IEnumerable<T> collection)
      {
         if(collection == null)
         {
            throw new ArgumentNullException("collection");
         }
         IList<T> list = new List<T>();
         foreach(T item in collection)
         {
            list.Add(item);
         }
         return list;
      }

      /// <summary>
      /// Zwraca wszystkie elementy kolekcji zgodne z przekazanym predykatem
      /// </summary>
      /// <typeparam name="T"></typeparam>
      /// <param name="iterator"></param>
      /// <param name="match"></param>
      /// <returns></returns>
      public static bool All<T>(IEnumerable<T> collection,Predicate<T> match)
      {
         if(collection == null)
         {
            throw new ArgumentNullException("collection");
         }
         if(match == null)
         {
            throw new ArgumentNullException("match");
         }
         foreach(T item in collection)
         {
            if(match(item) == false)
            {
               return false;
            }
         }
         return true;
      }
      /// <summary>
      /// Konwertuje wszystkie elementy kolekcji obiektowej typu T na nową tablicę typu U (zależnie od przekazanego konwertera)
      /// </summary>
      /// <typeparam name="T"></typeparam>
      /// <param name="collection"></param>
      /// <returns></returns>
      public static U[] UnsafeToArray<T,U>(IEnumerable collection,Converter<T,U> converter)
      {
         if(collection == null)
         {
            throw new ArgumentNullException("collection");
         }
         if(converter == null)
         {
            throw new ArgumentNullException("converter");
         }
         IEnumerable<U> newCollection = UnsafeConvertAll(collection,converter);
         return ToArray(newCollection);
      }

      /// <summary>
      /// Konwertuje wszystkie elementy iteratora obiektowego typu T na nową tablicę typu U (zależnie od przekazanego konwertera)
      /// </summary>
      /// <typeparam name="T"></typeparam>
      /// <param name="iterator"></param>
      /// <returns></returns>
      public static U[] UnsafeToArray<T,U>(IEnumerator iterator,Converter<T,U> converter)
      {
         if(iterator == null)
         {
            throw new ArgumentNullException("iterator");
         }
         if(converter == null)
         {
            throw new ArgumentNullException("converter");
         }
         IEnumerator<U> newIterator = UnsafeConvertAll(iterator,converter);
         return ToArray(newIterator);
      }

      /// <summary>
      /// Konwertuje kolekcję na tablicę
      /// </summary>
      /// <typeparam name="T"></typeparam>
      /// <param name="collection"></param>
      /// <returns></returns>
      public static T[] UnsafeToArray<T>(IEnumerable collection)
      {
         if(collection == null)
         {
            throw new ArgumentNullException("collection");
         }
         IEnumerator iterator = collection.GetEnumerator();

         using(iterator as IDisposable)
         {
            return UnsafeToArray<T>(iterator);
         }
      }
      /// <summary>
      /// Konwertuje iterator na tablicę
      /// </summary>
      /// <typeparam name="T"></typeparam>
      /// <param name="iterator"></param>
      /// <returns></returns>
      public static T[] UnsafeToArray<T>(IEnumerator iterator)
      {
         if(iterator == null)
         {
            throw new ArgumentNullException("iterator");
         }
         Converter<object,T> innerConverter = delegate(object item)
                                              {
                                                 return (T)item;
                                              };
         return UnsafeToArray(iterator,innerConverter);
      }
      /// <summary>
      /// Konwertuje wszystkie elementy typu T w kolekcji obiektowej na nową kolekcję typu U (zależnie od użytego konwertera)
      /// </summary>
      /// <typeparam name="T"></typeparam>
      /// <typeparam name="U"></typeparam>
      /// <param name="collection"></param>
      /// <param name="converter"></param>
      /// <returns></returns>
      public static IEnumerable<U> UnsafeConvertAll<T,U>(IEnumerable collection,Converter<T,U> converter)
      {
         if(collection == null)
         {
            throw new ArgumentNullException("collection");
         }
         if(converter == null)
         {
            throw new ArgumentNullException("converter");
         }
         foreach(object item in collection)
         {
            yield return converter((T)item);
         }
      }
      /// <summary>
      /// Konwertuje wszystkie elementy typu T w iteratorze typu IEnumerator na nową kolekcję typu U (zależnie od użytego konwertera)
      /// </summary>
      /// <typeparam name="T"></typeparam>
      /// <typeparam name="U"></typeparam>
      /// <param name="collection"></param>
      /// <param name="converter"></param>
      /// <returns></returns>
      public static IEnumerator<U> UnsafeConvertAll<T,U>(IEnumerator iterator,Converter<T,U> converter)
      {
         if(iterator == null)
         {
            throw new ArgumentNullException("collection");
         }
         if(converter == null)
         {
            throw new ArgumentNullException("converter");
         }
         while(iterator.MoveNext())
         {
            yield return converter((T)iterator.Current);
         }
      }
   }
}